function [sys,E]=Morse(Ep0,x0,a,m,varargin)
% Etats confins d'un atome hydrognode
%   >> Ep0          = Profondeur (eV) du puits
%   >> x0           = Abscisse du fond du puits
%   >> a            = Constante caractristique (m^{-1})
%   >> m            = Masse (par rapport  l'lectron) de l'objet physique
%   >> varargin     = Arguments d'entre optionnels
%           >> nx   = Nombre de point selon Ox (dfaut : 200)
%           >> nE   = Nombre de valeurs de l'nergie (dfaut : 1000)
%           >> Emin = Valeur minimale de l'nergie (dfaut : min(Ep))
%           >> Emax = Valeur maximale de l'nergie (dfaut : 0)
%
% Olivier Pujol : septembre 2013

%% Vrifications

if (Ep0<=0),error('Il faut Ep0 > 0');end
if (x0<=0),error('Il faut x0 > 0');end
if (m<=0),error('Il faut m > 0');end

%% Arguments d'entre optionnels

switch length(varargin)
    case 0
        nx=200;                     % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 1
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 2
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
    case 4
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
        E=linspace(varargin{3},varargin{4},nE);              % Ensemble des valeurs de l'nergie
    otherwise
        ('Mauvais nombre d''arguments d''entre optionnels');
end

%% Dtermination des limites du puits

% En thorie, l'nergie potentielle est infinie en x=0. Pour reprsenter cet infini, 
% on choisit la limite infrieure du puits de sorte que Ep = n*Ep0 (n=10).
% L'utilisateur peut changer cette valeur. 
n=10;
xA=x0-(1/a)*log(1+(n)^0.5);
clear n

% En thorie, l'nergie potentielle est nulle si tend vers l'infini. Pour reprsenter cet infini, 
% on choisit la limite suprieure du puits de sorte que Ep = epsilon*Ep0 (epsilon=0.001).
% L'utilisateur peut changer cette valeur. 
epsilon = 0.001;
xB=x0-(1/a)*log(1-(1-epsilon)^0.5);
clear epsilon

if (xA>=x0),error('La limite infrieure du puits est suprieure  x0');end
if (xB<=x0),error('La limite suprieure du puits est infrieure  x0');end
if (xA>=xB),error('La limite infrieure du puits est suprieure  sa limite suprieure');end

%% Energie potentielle

% Construction
x=linspace(xA,xB,nx);                               % Domaine spatial
Epc=Ep0*(1-exp(-a*(x-x0))).^2;       % Energie potentielle de Morse (Morse 1929) avec origine au fond du puits (cf. cours)

% Discrtisation
[M,xt,Ep]=discretisation(x,Epc); 
axis([0 xB 0 Ep0]);            % Pour avoir une bonne reprsentation graphique

% Valeurs de l'nergie
if (length(varargin)~=4)
    E=linspace(min(Ep),Ep0,nE);   % Ensemble des valeurs de l'nergie
end    

if (min(E)<0 || max(E)>Ep0),error('Certaines valeurs de l''nergie sont hors du domaine des tats confins');end

%% Etats confins

[sys,E]=Epot(M,xt,Ep,m,E);          % Fabrication du systme 
sysconf(sys,E);                     % Etats confins

return;